/*
 * Splits a single "vmtrace" log file into per-<pid> files.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include "list.h"
#define BITS_PER_LONG 32
#include "hash.h"
#include "vmtrace.h"


/* identify an open <pid> output file */
struct pid_file {
	unsigned int pid;
	int open_fd;
	struct list_head hash;
	char *filename;
};

char *output_dir;

struct list_head *hash_table;
int hash_bits = 12;	/* 4096 hash buckets */

int init_hash_table(void)
{
	int i;
	int hash_entries = 1 << hash_bits;

	hash_table = (struct list_head *) malloc(sizeof(struct list_head) *hash_entries);

	printf("hash_size: %d bytes\n", sizeof(struct list_head) * hash_entries);

	if (!hash_table) {
		printf("failed to malloc hash table!\n");
		return -1;
	}
	for (i=0;i<hash_entries;i++)
		INIT_LIST_HEAD(&hash_table[i]);

	return 0;
}

int insert_into_cache(struct pid_file *pidf)
{
	int bucket = hash_long(pidf->pid, hash_bits);
	list_add(&pidf->hash, &hash_table[bucket]);
}

struct pid_file* lookup_into_cache (unsigned long pid)
{
	struct list_head *entry;
	int bucket = hash_long(pid, hash_bits);

	list_for_each(entry, &hash_table[bucket]) {
		struct pid_file *pidf;
		pidf = list_entry(entry, struct pid_file, hash);
		if (pidf->pid == pid)
			return pidf;
	}
	return NULL;
}

struct pid_file *create_openfile(unsigned int pid)
{
	struct pid_file *pidf;
	int fd;
	char pathname[64];

	snprintf(pathname, sizeof(pathname), "%d.dat", pid);

	printf("%s\n", pathname);

	if ((fd = open(pathname, O_RDWR|O_CREAT)) < 0) {
		perror("open");
		printf("pathname: %s\n", pathname);
	}

	pidf = (struct pid_file*)malloc(sizeof(struct pid_file));

	if (!pidf) {
		printf("pidf allocation failure!");
		exit(0);
	}

	pidf->open_fd = fd;
	pidf->pid = pid;
	INIT_LIST_HEAD(&pidf->hash);
	pidf->filename = (char *)malloc(strlen(pathname));
	if (!pidf->filename) {
		printf("pidf filename allocation failure!\n");
		exit(0);
	}
	strcpy(pidf->filename, pathname);
	insert_into_cache(pidf);
	return pidf;
}

int write_entry(struct pid_file *pidf, struct vm_trace_entry *entry)
{
	int ret;

	ret = write(pidf->open_fd, entry, sizeof(struct vm_trace_entry));

	if (ret < 0) {
		printf("error writing to %s\n", pidf->filename);
		perror("write");
		return 0;
	}
	return 1;
}

int parse_trace_file(void *mptr, int len)
{
	int i;
	struct vm_trace_entry *entry = (struct vm_trace_entry *) mptr;

	for(i = 0; i < len/sizeof(struct vm_trace_entry); i++) {
		struct pid_file *pidf;

		pidf = lookup_into_cache(entry->pid);
		if (!pidf)
			pidf = create_openfile(entry->pid);
		if (!pidf) {
			printf("failure to create output pid file!\n");
			exit(0);
		}

		if (!write_entry(pidf, entry))
			exit(0);
		entry++;
	}
}


int create_dir(char *dir)
{
	int fd, ret, serrno;

	ret = mkdir(dir, S_IXUSR|S_IWUSR|S_IRUSR);

	serrno = errno;

	if (ret < 0 && serrno != EEXIST) {
		perror("mkdir");
		return -1;
	}

	output_dir = (char *)malloc(strlen(dir));
	if (!output_dir) {
		printf("create_dir malloc failure!\n");
		return -1;
	}

	strcpy(output_dir, dir);

	printf("output_dir:%s\n", output_dir);
}

int main(int argc, char *argv[])
{
	int fd;
	void *mptr;
	struct stat fstat;

	if (argc != 3) {
		printf("usage: vmtrace-split trace-file directory\n");
		exit(0);
	}

	if (stat(argv[1], &fstat) < 0) {
		perror("stat");
		exit(0);
	}

	if ((fd = open(argv[1], O_RDONLY)) < 0) {
		perror("open");
		exit(0);
	}

	mptr = mmap(0, fstat.st_size, PROT_READ, MAP_SHARED, fd, 0);

	if (mptr == MAP_FAILED) {
		perror("mmap");
		exit(0);
	}

	if (create_dir(argv[2]) < 0) {
		printf("creation of directory failed!\n");
		exit(0);
	}

	if (init_hash_table() < 0) {
		printf("creation of hashtable failed!\n");
		exit(0);
	}

	if (chdir(argv[2]) < 0) {
		perror("chdir");
		exit(0);
	}

	parse_trace_file(mptr, fstat.st_size);

	munmap(mptr, fstat.st_size);
	close(fd);
}

